#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <opencv2/opencv.hpp>
#include "Percipio/DllApi/TYAppDllApi.h"
#include "PMDefines.h"

static char gErrMsg[512];

#define ASSERT_OK(x) do{ \
    printf(">>> " #x "\n"); \
    TY_STATUS err = (x); \
    if(err){ \
      TYAppLastError(&err, gErrMsg, sizeof(gErrMsg)); \
      printf(#x " ret error: (%d)%s\n", err, gErrMsg); \
      abort(); \
    } \
  }while(0)


class FakeLock
{
public:
  FakeLock() : _lock(0) {}
  void lock() {
    while(_lock) {cv::waitKey(1);}
    _lock = 1;
  }
  void unlock() {
    _lock = 0;
  }
private:
  bool _lock;
};

struct DevImg {
  std::string sn;
  bool isDepth;
  cv::Mat image;
};

struct AllData
{
  FakeLock lock;

  std::queue<DevImg> imgQ;

  bool    exit;

  cv::Mat depth;
  cv::Mat color;

  cv::Vec3i boxSize;
  std::vector< std::vector<cv::Point2i> > bounding;
};


// void TY_STDC Event_Cb(int32_t id, const char* msg, void* user_data)
void TY_STDC Event_Cb(const XData* head, const void* data, void* user_data)
{
  if(head->error_id){
    printf("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
    printf("!!! GOT ERROR: %d(%s)\n", head->error_id, (const char*)data);
    printf("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
  } else {
    printf("got event: act(%d) tar(0x%x) uid(%d) len(%d)"
        , head->action_id, head->target_id
        , head->uid, head->length);
    if(head->target_id == I_CMD_RESET_BG_IMAGE){
        int32_t left = *(const int*)data;
        printf("Reset bg need %d images more\n", left);
    }
  }
}

void Depth_Cb(const BlockHeader* result, int32_t blockSize, void* user_data)
{
  ImageHeader* img = (ImageHeader*)result;
//  printf("Depth: format(%d) pixelType(%d) pixelSize(%d) width(%d) height(%d) size(%d)\n"
//      , img->format, img->pixelType, img->pixelSize, img->width, img->height, img->size);
  assert( img->format == I_IMG_FORMAT_RAW );

  AllData* p = (AllData*)user_data;

  p->lock.lock();

  DevImg out;
  out.sn = img->devSN;
  out.isDepth = 1;
  out.image = cv::Mat(img->height, img->width, CV_16U, &img[1]).clone();
  p->imgQ.push(out);

  p->lock.unlock();
}

void Color_Cb(const BlockHeader* result, int32_t blockSize, void* user_data)
{
  ImageHeader* img = (ImageHeader*)result;
//  printf("Color: format(%d) pixelType(%d) pixelSize(%d) width(%d) height(%d) size(%d)\n"
//      , img->format, img->pixelType, img->pixelSize, img->width, img->height, img->size);
  assert( img->format == I_IMG_FORMAT_JPG );

  AllData* p = (AllData*)user_data;

  p->lock.lock();

  DevImg out;
  out.sn = img->devSN;
  out.isDepth = 0;
  cv::Mat jpeg(img->height, img->width, CV_8UC1, &img[1]);
  out.image = cv::imdecode(jpeg,CV_LOAD_IMAGE_COLOR);
  p->imgQ.push(out);

  p->lock.unlock();
}

void P3d_Cb(const BlockHeader* result, int32_t blockSize, void* user_data)
{
  ImageHeader* img = (ImageHeader*)result;
  printf("P3d: format(%d) pixelType(%d) pixelSize(%d) width(%d) height(%d) size(%d)\n"
      , img->format, img->pixelType, img->pixelSize, img->width, img->height, img->size);
}

void PM_Cb(const BlockHeader* result, int32_t blockSize, void* user_data)
{
  const PackageInfo* pm = (PackageInfo*)result;
  printf("PM: type(%d) size(%f,%f,%f)\n", pm->type, pm->sizeX, pm->sizeY, pm->sizeZ);

  AllData* p = (AllData*)user_data;
  p->boxSize = cv::Vec3i(pm->sizeX, pm->sizeY, pm->sizeZ);
  p->bounding[0][0] = cv::Point2i(pm->pixelPoints[0], pm->pixelPoints[1]);
  p->bounding[0][1] = cv::Point2i(pm->pixelPoints[2], pm->pixelPoints[3]);
  p->bounding[0][2] = cv::Point2i(pm->pixelPoints[4], pm->pixelPoints[5]);
  p->bounding[0][3] = cv::Point2i(pm->pixelPoints[6], pm->pixelPoints[7]);
  p->bounding[1][0] = cv::Point2i(pm->pixelPointsRGB[0], pm->pixelPointsRGB[1]);
  p->bounding[1][1] = cv::Point2i(pm->pixelPointsRGB[2], pm->pixelPointsRGB[3]);
  p->bounding[1][2] = cv::Point2i(pm->pixelPointsRGB[4], pm->pixelPointsRGB[5]);
  p->bounding[1][3] = cv::Point2i(pm->pixelPointsRGB[6], pm->pixelPointsRGB[7]);
}

void TY_STDC Data_Cb(const BlockHeader* result, int32_t blockSize, void* user_data)
{
  switch(result->dataType) {
    case I_DATA_DEPTH_IMAGE: Depth_Cb(result, blockSize, user_data); break;
    case I_DATA_COLOR_IMAGE: Color_Cb(result, blockSize, user_data); break;
    case I_DATA_PACKAGE_MEASURE: PM_Cb(result, blockSize, user_data); break;
    case I_DATA_P3D: P3d_Cb(result, blockSize, user_data); break;
    default:
      printf("unknown data type(%d)\n", result->dataType);
  }
}

int main(int argc, const char* argv[])
{
  TY_STATUS err;
  int32_t n;

  AllData data;
  data.exit = 0;
  data.bounding.resize(2);
  data.bounding[0].resize(4);
  data.bounding[1].resize(4);
  data.depth = cv::Mat::zeros(10,10,CV_16U);
  data.color = cv::Mat::zeros(10,10,CV_8UC3);

  // for debug
  ASSERT_OK( TYAppInit(argc, argv) );
  ASSERT_OK( TYAppDeinit() );

  ASSERT_OK( TYAppInit(argc, argv) );
  ASSERT_OK( TYAppSetDataCallback(Data_Cb, &data) );
  ASSERT_OK( TYAppSetEventCallback(Event_Cb, &data) );

  bool b = 1;
  // enable depth image
  ASSERT_OK( TYAppWriteProperty(I_PROPERTY_bool_GRAB_DEPTH, &b, sizeof(b)) );
  // enable color image
  ASSERT_OK( TYAppWriteProperty(I_PROPERTY_bool_GRAB_COLOR, &b, sizeof(b)) );
  // enable pm
  ASSERT_OK( TYAppWriteProperty(I_PROPERTY_bool_GRAB_PM, &b, sizeof(b)) );

  // set depth image to RAW format
  n = I_IMG_FORMAT_RAW;
  ASSERT_OK( TYAppWriteProperty(I_PROPERTY_int_DEPTH_FORMAT, &n, sizeof(n)) );
  // set color image to JPG format
  n = I_IMG_FORMAT_JPG;
  ASSERT_OK( TYAppWriteProperty(I_PROPERTY_int_COLOR_FORMAT, &n, sizeof(n)) );

  // set bg rect
  cv::Rect bgRect(170,130, 300, 220);
  ASSERT_OK( TYAppWriteProperty(I_PROPERTY_int4_DEPTH_ROI, &bgRect, sizeof(bgRect)) );
  
  // set safe rect
  cv::Rect safeRect(160,120, 320, 240);
  ASSERT_OK( TYAppWriteProperty(I_PROPERTY_int4_SAFE_RECT, &safeRect, sizeof(safeRect)) );

  // get bg rect
  ASSERT_OK( TYAppReadProperty(I_PROPERTY_int4_DEPTH_ROI, &bgRect, sizeof(bgRect), NULL) );
  
  // get safe rect
  ASSERT_OK( TYAppReadProperty(I_PROPERTY_int4_SAFE_RECT, &safeRect, sizeof(safeRect), NULL) );

  cv::Mat depth8U;
  cv::Mat depthShow(480, 640, CV_16U);

  ASSERT_OK( TYAppStart() );

  printf("====================================\n");
  printf(">>> press keys to\n");
  printf(" 'c' : reset bg images\n");
  printf(" 'd' : calculate once\n");
  printf(" 't' : trigger once(only work in trigger mode)\n");
  printf(" 'q' : quit\n");
  printf("====================================\n");
        // ASSERT_OK( TYAppWriteCmd(I_CMD_RESET_BG_IMAGE) );
        // ASSERT_OK( TYAppCalcOnce() );
  while(!data.exit)
  {
    DevImg img;

    data.lock.lock();

    if(!data.imgQ.size()){
      data.lock.unlock();
    } else {
      img = data.imgQ.front();
      data.imgQ.pop();

      data.lock.unlock();

      if(img.isDepth){
        cv::Mat(img.image / 8).convertTo(depth8U, CV_8U);
        cv::cvtColor(depth8U, depthShow, cv::COLOR_GRAY2BGR);
        cv::rectangle(depthShow, bgRect, cv::Scalar(0,255,0));
        cv::rectangle(depthShow, safeRect, cv::Scalar(255,0,0));
        char text[64];
        sprintf(text, "box: %d %d %d", data.boxSize[0], data.boxSize[1], data.boxSize[2]);
        cv::putText(depthShow, text, cv::Point(0,20), cv::FONT_HERSHEY_SIMPLEX, 0.5, cv::Scalar(0,255,0), 2);
        cv::drawContours(depthShow, data.bounding, 0, cv::Scalar(0,255,0));
        imshow("depth-" + img.sn, depthShow);
      } else {
        imshow("color-" + img.sn, img.image);
      }
    }

    int key = cv::waitKey(10) & 0xff;
    switch(key){
      case 'c':
        ASSERT_OK( TYAppWriteCmd(I_CMD_RESET_BG_IMAGE) );
        break;
      case 'd':
        ASSERT_OK( TYAppCalcOnce() );
        break;
      case 't':
        ASSERT_OK( TYAppCalcOnce() );
        break;
      case 'q':
        data.exit = 1;
        break;
    }
  }

  ASSERT_OK( TYAppStop() );

  ASSERT_OK( TYAppDeinit() );
  return 0;
}
